import { QueryConfig } from "@/lib/react-query";
import { Task } from "@/lib/types/task";
import { createClient } from "@/utils/supabase/client";
import { supabaseConfig } from "@/utils/supabase/config";
import { queryOptions, useQuery } from "@tanstack/react-query";

export const getUserTask = async (): Promise<Task[]> => {
  const supabase = createClient();
  const supabaseUser = await supabase.auth.getUser();
  const userId = supabaseUser.data.user?.id;
  if (!userId) {
    throw new Error("User not found");
  }

  const { data, error } = await supabase
    .from(supabaseConfig.column.task)
    .select(
      `
        id,
        created_at,
        title,
        status,
        due_date,
        priority,
        user ( id, name, email, role )
    `
    )
    .eq("assignee_id", userId)
    .overrideTypes<Task[]>();
  if (error) {
    throw new Error(error.message);
  }
  const tasks = data as Task[];
  return tasks.map((task) => ({
    ...task,
    created_at: new Date(task.created_at),
  }));
};

export const getUserTaskQueryOptions = () => {
  return queryOptions({
    queryKey: ["user-task"],
    queryFn: getUserTask,
  });
};

type UseGetUserTaskOptions = {
  queryConfig?: QueryConfig<typeof getUserTask>;
};

export const useGetUserTask = ({ queryConfig }: UseGetUserTaskOptions = {}) => {
  return useQuery({
    ...getUserTaskQueryOptions(),
    ...queryConfig,
  });
};
