/* eslint-disable  @typescript-eslint/no-explicit-any */

import { Task } from "./task";
import { User } from "./user";
import { TaskPriorityEnum, TaskStatusEnum } from "../enums/task";

export type TColumn = "pending" | "inprogress" | "completed";

export function enumToTColumn(status: TaskStatusEnum): TColumn {
  switch (status) {
    case TaskStatusEnum.PENDING:
      return "pending";
    case TaskStatusEnum.IN_PROGRESS:
      return "inprogress";
    case TaskStatusEnum.COMPLETED:
      return "completed";
    default:
      throw new Error(`Invalid status: ${status}`);
  }
}

export type TColumnProps = {
  title: string;
  column: TColumn;
  cards: TCard[];
  insertCard: (card: TCard) => void;
  updateCard: (card: TCard) => void;
  deleteCard: (card: TCard) => void;
  className?: string;
};

export function statusToTColumn(status: string): TColumn {
  switch (status) {
    case "Pending":
      return "pending";
    case "In Progress":
      return "inprogress";
    case "Completed":
      return "completed";
    default:
      throw new Error(`Invalid status: ${status}`);
  }
}

export function tcolumnToEnum(t: TColumn): TaskStatusEnum {
  switch (t) {
    case "pending":
      return TaskStatusEnum.PENDING;
    case "inprogress":
      return TaskStatusEnum.IN_PROGRESS;
    case "completed":
      return TaskStatusEnum.COMPLETED;
    default:
      return TaskStatusEnum.PENDING;
  }
}

export type TCard = {
  id: string;
  created_at: Date;
  title: string;
  due_date: Date | null;
  priority: TaskPriorityEnum;
  user: User | null;
} & Pick<TColumnProps, "column">;

export function tasksToTCards(tasks: Task[]): TCard[] {
  return tasks.map(
    (task): TCard => ({
      id: task.id,
      created_at: task.created_at,
      title: task.title,
      due_date: task.due_date,
      priority: task.priority,
      column: statusToTColumn(task.status),
      user: task.user,
    })
  );
}

export function tcardToTask(card: TCard): Task {
  return {
    id: card.id,
    created_at: card.created_at,
    title: card.title,
    status: tcolumnToEnum(card.column),
    due_date: card.due_date,
    priority: card.priority,
    user: card.user,
  };
}

export type TCardProps = {
  card: TCard;
  hanldeDragStart: (event: any, card: TCard) => void;
} & Pick<TColumnProps, "updateCard" | "deleteCard">;

type ExtractIdType<T> = T extends { id: infer U } ? U : never;

export type TDropsIndicatorProps = {
  beforeId: ExtractIdType<TCard>;
} & Pick<TColumnProps, "column">;

export type TBurnBarrelProps = Pick<TColumnProps, "deleteCard">;

export type TAddCardProps = Pick<TColumnProps, "insertCard" | "column">;
