import {
  queryOptions,
  useMutation,
  useQuery,
  useQueryClient,
} from "@tanstack/react-query";
import { createClient } from "@/utils/supabase/client";
import { User, UserDto } from "./types/user";
import { LoginInput, RegisterInput } from "./validations/auth";
import { toRoleEnum, UserRoleEnum } from "./enums/role";
import { getMembers } from "@/features/user/supabase/get-members";
import { supabaseConfig } from "@/utils/supabase/config";

export const getUser = async (): Promise<User> => {
  const supabase = createClient();
  const supabaseUser = await supabase.auth.getUser();

  const response = await supabase
    .from("user")
    .select("*")
    .eq("id", supabaseUser.data.user?.id)
    .single<UserDto>();

  const userDto = response.data;

  if (!userDto) {
    throw new Error("User not found");
  }

  const user: User = {
    id: userDto.id,
    email: userDto.email,
    name: userDto.name,
    role: toRoleEnum(userDto.role),
    position: userDto.position,
  };

  return user;
};

const userQueryKey = ["user"];

export const getUserQueryOptions = () => {
  return queryOptions({
    queryKey: userQueryKey,
    queryFn: getUser,
  });
};

export const useUser = () => useQuery(getUserQueryOptions());

export const useLogin = ({
  onSuccess,
  onError,
}: {
  onSuccess?: () => void;
  onError?: (error: Error) => void;
}) => {
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: loginWithEmailAndPassword,
    onSuccess: (data) => {
      queryClient.setQueryData(userQueryKey, data);
      onSuccess?.();
    },
    onError: (error) => {
      onError?.(error);
    },
  });
};

export const useRegister = ({
  onSuccess,
  onError,
}: {
  onSuccess?: () => void;
  onError?: (error: Error) => void;
}) => {
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: register,
    onSuccess: (data) => {
      queryClient.setQueryData(userQueryKey, data);
      onSuccess?.();
    },
    onError: (error) => {
      onError?.(error);
    },
  });
};

export const useLogout = ({
  onSuccess,
  onError,
}: {
  onSuccess?: () => void;
  onError?: (error: Error) => void;
}) => {
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: logout,
    onSuccess: () => {
      queryClient.removeQueries({ queryKey: userQueryKey });
      onSuccess?.();
    },
    onError: (error) => {
      onError?.(error);
    },
  });
};

const logout = async (): Promise<void> => {
  const supabase = createClient();
  await supabase.auth.signOut();
};

const loginWithEmailAndPassword = async (input: LoginInput): Promise<User> => {
  const supabase = createClient();
  const { error } = await supabase.auth.signInWithPassword({
    email: input.email,
    password: input.password,
  });
  if (error) {
    throw new Error(error.message);
  }
  const user = await getUser();
  return user;
};

const register = async (input: RegisterInput): Promise<User> => {
  const supabase = createClient();
  const { data, error } = await supabase.auth.signUp({
    email: input.email,
    password: input.password,
  });
  if (error) {
    throw new Error(error.message);
  }
  if (!data.user) {
    throw new Error("Failed to register user");
  }
  const user: User = {
    id: data.user.id,
    email: input.email,
    name: null,
    role: null,
    position: null,
  };

  await insertUser(user);

  return user;
};

const insertUser = async (data: User): Promise<void> => {
  let position = null;
  if (data.role == UserRoleEnum.MEMBER) {
    const members = await getMembers();
    position = members.length + 1;
  }
  const supabase = createClient();
  const { error } = await supabase.from(supabaseConfig.column.user).insert({
    id: data.id,
    email: data.email,
    name: data.name,
    role: data.role,
    position: position,
  });
  if (error) {
    throw new Error(error.message);
  }
  const currentUser = await getUser();
  const id = currentUser.id;
  if (!id) {
    throw new Error("Failed to register user to database");
  }
};
