import { UserRoleEnum } from "@/lib/enums/role";
import { MutationConfig } from "@/lib/react-query";
import { User, UserDto } from "@/lib/types/user";
import { UpdateUserInput } from "@/lib/validations/user";
import { createClient } from "@/utils/supabase/client";
import { supabaseConfig } from "@/utils/supabase/config";
import { useMutation } from "@tanstack/react-query";
import { getMembers } from "./get-members";

export const updateUserPosition = async ({
  data,
}: {
  data: UpdateUserInput;
}): Promise<User[]> => {
  if (!data.id) {
    throw new Error("User id is required");
  }

  if ((data.position ?? 0) < 0) {
    throw new Error("Position must be greater than 0");
  }

  const supabase = createClient(); //  You should have your Supabase client initialized.
  const members = (await getMembers()).filter(
    (member) => member.position !== null
  );
  const oldMember = members.find((member) => member.id === data.id);

  if (!oldMember) {
    throw new Error("User not found");
  }

  const newMember = {
    ...oldMember,
    position: data.position,
  } as User;

  const copyMembers = [...members].filter((member) => member.id !== data.id);

  if (oldMember.position === newMember.position) return members;
  var shift = 1;
  if (oldMember.position! < newMember.position!) {
    shift = 2;
  }
  if (newMember.position == members.length) {
    shift = 1;
  }

  copyMembers.splice(newMember.position! - shift, 0, newMember);

  const updatedMembers = copyMembers.map((member, index) => {
    return {
      ...member,
      position: index + 1,
    } as User;
  });

  updatedMembers.forEach(async (member) => {
    const { error } = await supabase
      .from(supabaseConfig.column.user)
      .update({
        position: member.position,
      })
      .eq("id", member.id);
    if (error) {
      throw new Error(error.message);
    }
  });

  return updatedMembers;
};

type UseUpdateUserPositionOptions = {
  mutationConfig?: MutationConfig<typeof updateUserPosition>;
};

export const useUpdateUserPosition = ({
  mutationConfig,
}: UseUpdateUserPositionOptions) => {
  const { onSuccess, onError, ...restConfig } = mutationConfig || {};

  return useMutation({
    onSuccess: (...args) => {
      onSuccess?.(...args);
    },
    onError: (...args) => {
      onError?.(...args);
    },
    ...restConfig,
    mutationFn: updateUserPosition,
  });
};
