import { TOwnerCardProps } from "@/lib/types/owner-kanban";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { useState } from "react";
import { format, differenceInDays } from "date-fns";

export default function OwnerBoardCard({ card }: TOwnerCardProps) {
  const [isDialogOpen, setIsDialogOpen] = useState(false);

  return (
    <>
      <div
        className="rounded border border-neutral-700 bg-neutral-800 p-5 lg:p-3 my-2 relative cursor-pointer"
        onClick={() => setIsDialogOpen(true)}
      >
        <div className="flex items-center gap-2 mb-2 flex-wrap">
          <span
            className={`px-2 py-0.5 text-xs rounded ${
              {
                pending: "bg-yellow-600/20 text-yellow-400",
                inprogress: "bg-blue-600/20 text-blue-400",
                completed: "bg-green-600/20 text-green-400",
              }[card.status.toLowerCase()] ||
              "bg-neutral-600/20 text-neutral-400"
            }`}
          >
            {card.status}
          </span>
          <span
            className={`px-2 py-0.5 text-xs rounded ${
              {
                high: "bg-red-600/20 text-red-400",
                medium: "bg-orange-600/20 text-orange-400",
                low: "bg-green-600/20 text-green-400",
              }[card.priority.toLowerCase()] ||
              "bg-neutral-600/20 text-neutral-400"
            }`}
          >
            {card.priority}
          </span>
          <div className="flex-grow"></div>
          {card.due_date &&
            (() => {
              const dueDate = new Date(card.due_date);
              const today = new Date();
              const daysRemaining = differenceInDays(dueDate, today);
              const dueDateStyles = {
                overdue: "bg-red-600/30 text-red-400",
                urgent: "bg-amber-600/30 text-amber-400",
                warning: "bg-orange-600/30 text-orange-400",
                normal: "bg-blue-600/30 text-blue-400",
              };
              const style =
                daysRemaining < 0
                  ? dueDateStyles.overdue
                  : daysRemaining === 0
                  ? dueDateStyles.urgent
                  : daysRemaining <= 3
                  ? dueDateStyles.warning
                  : dueDateStyles.normal;
              return (
                <span className={`px-2 py-0.5 text-xs rounded ${style}`}>
                  Due {format(dueDate, "MMM d")}{" "}
                  {daysRemaining >= 0 && `(${daysRemaining}d)`}
                </span>
              );
            })()}
        </div>
        <p className="text-sm text-neutral-100 break-words">{card.title}</p>

        <div className="flex flex-row">
          <div className="flex-1"></div>
          <span className={`text-xs text-neutral-100  rounded`}>
            {format(card.created_at, "MMM dd, HH:mm")}
          </span>
        </div>
      </div>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Task Details</DialogTitle>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <h4 className="text-sm font-medium text-neutral-950">Title</h4>
              <p className="text-neutral-950">{card.title}</p>
            </div>
            <div>
              <h4 className="text-sm font-medium text-neutral-950">Status</h4>
              <p className="text-neutral-950">{card.status}</p>
            </div>
            <div>
              <h4 className="text-sm font-medium text-neutral-950">Priority</h4>
              <p className="text-neutral-950">{card.priority}</p>
            </div>
            <div>
              <h4 className="text-sm font-medium text-neutral-950">Due Date</h4>
              <p className="text-neutral-950">
                {card.due_date
                  ? format(new Date(card.due_date), "PPP")
                  : "No due date"}
              </p>
            </div>
            <div>
              <h4 className="text-sm font-medium text-neutral-950">
                Created At
              </h4>
              <p className="text-neutral-950">
                {format(new Date(card.created_at), "PPP")}
              </p>
            </div>
            {card.user && (
              <div>
                <h4 className="text-sm font-medium text-neutral-950">
                  Assigned To
                </h4>
                <p className="text-neutral-950">
                  {card.user.name || card.user.email}
                </p>
              </div>
            )}
          </div>
        </DialogContent>
      </Dialog>
    </>
  );
}
