/* eslint-disable  @typescript-eslint/no-explicit-any */

"use client";

import { useEffect, useState } from "react";
import { useMembers } from "../user/supabase/get-members";
import { User } from "@/lib/types/user";
import { MemberDropIndicator } from "./member-drop-indicator";
import { MemberCard } from "./member-card";
import { updateUser, useUpdateUser } from "../user/supabase/update-user";
import { toast } from "sonner";
import { useUpdateUserPosition } from "../user/supabase/update-user-position";

export default function MemberList() {
  const members = useMembers();
  const updateUserPosition = useUpdateUserPosition({
    mutationConfig: {
      onSuccess: (data) => {
        toast.success("User position updated successfully");
      },
      onError: (error: Error) => {
        toast.error(error.message);
      },
    },
  });
  const [cards, setCards] = useState<User[]>([]);

  useEffect(() => {
    const data = members.data;

    if (data) {
      setCards(data);
    }
  }, [members.data != undefined]);

  function rearrangeCards(members: User[], newMember: User) {
    const oldMember = members.find((member) => member.id === newMember.id);
    if (!oldMember) return;
    if (oldMember.position === newMember.position) return;
    var shift = 1;
    if (oldMember.position! < newMember.position!) {
      shift = 2;
    }
    if (newMember.position == members.length) {
      shift = 1;
    }

    const copyMembers = [...members].filter(
      (member) => member.id !== newMember.id
    );
    console.log(newMember.position);
    copyMembers.splice(newMember.position! - shift, 0, newMember);

    const updatedMembers = copyMembers.map((member, index) => {
      return {
        ...member,
        position: index + 1,
      } as User;
    });
    setCards(updatedMembers);
  }

  const [active, setActive] = useState(false);

  const getInidicator = () => {
    return Array.from(
      document.querySelectorAll(`[data-column=user]`)
    ) as HTMLElement[];
  };
  const getNearestIndicator = (event: any, indicators?: HTMLElement[]) => {
    const DISTANCE_OFFSET = 50;
    const nearest = indicators?.reduce(
      (closest: any, child: Element) => {
        const box = child.getBoundingClientRect();
        const offset = event.clientY - (box.top + DISTANCE_OFFSET);
        if (offset < 0 && offset > closest.offset) {
          return { offset, element: child };
        } else {
          return closest;
        }
      },
      {
        offset: Number.NEGATIVE_INFINITY,
        element: indicators[indicators.length - 1],
      }
    );
    return nearest;
  };
  const clearHighLights = (elements?: HTMLElement[]) => {
    const indicators = elements || getInidicator();
    indicators?.forEach((element) => (element.style.opacity = "0"));
  };
  const highLightIndicator = (event: any) => {
    const indicators = getInidicator();
    clearHighLights(indicators);
    const nearest = getNearestIndicator(event, indicators);
    nearest.element.style.opacity = "1";
  };
  const handleDragStart = (
    event: { dataTransfer: { setData: (arg0: string, arg1: string) => void } },
    card: User
  ) => {
    event.dataTransfer.setData("cardId", card.id);
  };
  const handleDragOver = (event: { preventDefault: () => void }) => {
    event.preventDefault();
    highLightIndicator(event);
    setActive(true);
  };
  const handleDragLeave = () => {
    setActive(false);
    clearHighLights();
  };
  const handleDrop = (event: {
    dataTransfer: { getData: (arg0: string) => any };
  }) => {
    setActive(false);
    clearHighLights();
    const cardId = event.dataTransfer.getData("cardId");
    const indicators = getInidicator();
    const { element } = getNearestIndicator(event, indicators);
    const after = element.dataset.before;
    if (after !== cardId) {
      const card = cards.find((card) => card.id === cardId);
      const afterCard = cards.find((card) => card.id === after);

      console.log(card?.position, afterCard?.position);
      if (!card) {
        toast.error("Card not found");
        return;
      }
      if (!afterCard) {
        rearrangeCards(cards, {
          ...card,
          position: cards.length,
        });
        updateUserPosition.mutate({
          data: {
            ...card,
            role: card.role?.toString(),
            position: cards.length,
          },
        });
        return;
      }
      const afterCardIndex = cards.indexOf(afterCard);
      rearrangeCards(cards, {
        ...card,
        position: afterCardIndex + 1,
      });
      updateUserPosition.mutate({
        data: {
          ...card,
          role: card.role?.toString(),
          position: afterCardIndex + 1,
        },
      });
    }
  };
  return (
    <div className={`flex-1 mb-2.5 lg:mb-0`}>
      <div
        onDragOver={handleDragOver}
        onDragLeave={handleDragLeave}
        onDrop={handleDrop}
        className={`h-full w-full transition-colors
        ${active ? "bg-neutral-800/50" : "bg-neutral-800/0"}`}
      >
        {cards?.map((card) => (
          <MemberCard
            key={card?.id}
            card={card}
            handleDragStart={handleDragStart}
            setCards={setCards}
          />
        ))}
        <MemberDropIndicator beforeId="-1" />
      </div>
    </div>
  );
}
